#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
hmm9.py
HDGL lattice with audio, PHI-weighted P2P, Base4096 streaming, and real-time OpenGL folding.
Thread-safe: compute runs in OpenGL context, audio reads precomputed deltas, supports JSON/binary export and adjustable viewport scrolling.
"""

import sys, json, math, struct, unicodedata, hmac, hashlib, queue, threading, ctypes
import numpy as np
import sounddevice as sd
from base4096 import encode, decode
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 128
LATTICE_HEIGHT = 1024
CHUNK_HEIGHT = 256
THRESHOLD = np.sqrt(1.6180339887)
DELTA_THRESHOLD = 0.01
PHI = 1.6180339887
PHI_POWERS = np.array([1.0/pow(PHI,7*(i+1)) for i in range(72)], dtype=np.float32)
AUTO_SCROLL_SPEED = 1
SAMPLE_RATE = 44100
FRAME_BUFFER_SIZE = 1024
MAX_SLOTS = 16_777_216
RECURSION_DEPTH = 3

EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BINARY = "hdgl_lattice.hdgl"
EXPORT_BASE4096 = "vectors.b4096"
HMAC_KEY = b"ZCHG-Base4096-Signature-Key"

# Shared delta queue
delta_queue = queue.Queue(maxsize=4096)

# Globals
cycle = 0.0
omega_time = 0.0
compute_shader = None
lattice_tex = None
prev_tex = None
delta_ssbo = None
shader = None
current_yoffset = 0
manual_yoffset = None

# -------------------------------
# CHAR SLOT HELPERS
# -------------------------------
def is_valid_char(c):
    try:
        cp = ord(c)
        if 0xD800 <= cp <= 0xDFFF:
            return False
        name = unicodedata.name(c)
        cat = unicodedata.category(c)
        if any(bad in name for bad in ['CONTROL','PRIVATE USE','UNASSIGNED','TAG']):
            return False
        if cat in ['Mn','Mc','Me','Cc','Cf','Cs','Cn','Co','Zs']:
            return False
        return True
    except ValueError:
        return False

def hdgl_char(idx):
    h = (idx * 2654435761) % 0x110000
    c = chr(h) if is_valid_char(chr(h)) else chr((h+1)%0x110000)
    return c

# -------------------------------
# RECURSIVE VECTORS
# -------------------------------
def unfold_slot(idx, depth=0):
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {
        "idx": idx,
        "value": val,
        "char": hdgl_char(idx),
        "children": []
    }
    if depth < RECURSION_DEPTH:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def flatten_vector(v):
    if isinstance(v, (list, tuple)):
        return ''.join(flatten_vector(x) for x in v)
    if isinstance(v, dict):
        return ''.join(flatten_vector(v[k]) for k in ['char','value','children'] if k in v)
    return str(v)

def build_recursive_vectors(num_samples=32):
    return [unfold_slot(idx) for idx in range(num_samples)]

# -------------------------------
# BASE4096 SURROGATE-SAFE
# -------------------------------
def string_to_base4096_bytes(s):
    data = bytearray()
    for c in s:
        cp = ord(c)
        if 0xD800 <= cp <= 0xDFFF or cp > 0x10FFFF:
            continue
        data.extend(cp.to_bytes(3, byteorder='big', signed=False))
    return bytes(data)

def base4096_bytes_to_string(b):
    out = []
    for i in range(0, len(b), 3):
        chunk = int.from_bytes(b[i:i+3], 'big')
        if chunk >= 0x110000:
            raise ValueError(f"Invalid codepoint {chunk}")
        out.append(chr(chunk))
    return ''.join(out)

# -------------------------------
# EXPORT FUNCTIONS
# -------------------------------
def export_recursive_vectors_base4096(vectors, outfile=EXPORT_BASE4096, hmac_key=HMAC_KEY):
    """Streaming Base4096 export with HMAC chaining."""
    h = hmac.new(hmac_key, digestmod=hashlib.sha256)
    with open(outfile, "w", encoding="utf-8") as f:
        for vector in vectors:
            combined = flatten_vector(vector)
            data = string_to_base4096_bytes(combined)
            h.update(data)
            encoded = encode(data)
            f.write(encoded + "\n")
        sig_encoded = encode(h.digest())
        f.write(f"#HMAC:{sig_encoded}\n")
    print(f"✅ Exported vectors to Base4096 with embedded HMAC: {outfile}")

def export_recursive_vectors_json(vectors, outfile=EXPORT_JSON):
    def filter_surrogates(obj):
        if isinstance(obj, dict):
            return {k: filter_surrogates(v) for k,v in obj.items()}
        if isinstance(obj, list):
            return [filter_surrogates(x) for x in obj]
        if isinstance(obj, str):
            return ''.join(c for c in obj if 0xD800 > ord(c) or ord(c) > 0xDFFF)
        return obj
    safe_vectors = filter_surrogates(vectors)
    with open(outfile, "w", encoding="utf-8") as f:
        json.dump(safe_vectors, f, ensure_ascii=False, indent=2)
    print(f"✅ Exported {len(vectors)} vectors to JSON: {outfile}")

def export_binary_lattice(num_samples=LATTICE_WIDTH, outfile=EXPORT_BINARY):
    with open(outfile, "wb") as f:
        for idx in range(num_samples):
            val = (idx * 2654435761) % 4096 / 4096.0
            packed = struct.pack("fI", val, idx)
            f.write(packed)
    print(f"✅ Exported {num_samples} lattice slots to binary: {outfile}")

# -------------------------------
# P2P STUB
# -------------------------------
def propagate_to_peers(coords):
    """PHI-weighted delta propagation."""
    weighted = [(x, y, PHI**(-i%72)) for i, (x,y) in enumerate(coords)]
    # TODO: send `weighted` to connected peers
    pass

# -------------------------------
# SHADERS
# -------------------------------
COMPUTE_SRC = """
#version 430
layout(local_size_x=16, local_size_y=16) in;

layout(rgba32f, binding=0) uniform image2D latticeTex;
layout(rgba32f, binding=1) uniform image2D prevTex;

layout(std430, binding=2) buffer DeltaBuffer {
    uint count;
    ivec2 coords[];
};

uniform float cycle;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform float deltaThreshold;

float hdgl_slot(float val, float r_dim, float omega, int x, int y){
    float resonance = (x % 4 == 0 ? 0.1 * sin(cycle*0.05 + float(y)) : 0.0);
    float wave = (x % 3 == 0 ? 0.3 : (x % 3 == 1 ? 0.0 : -0.3));
    float omega_inst = phiPowers[y % 72];
    float rec = r_dim*val*0.5 + 0.25*sin(cycle*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val > threshold ? 1.0 : 0.0;
}

void main() {
    ivec2 coord = ivec2(gl_GlobalInvocationID.xy);
    ivec2 size = imageSize(latticeTex);
    if(coord.x >= size.x || coord.y >= size.y) return;

    vec4 prev = imageLoad(prevTex, coord);
    vec4 val = imageLoad(latticeTex, coord);

    float r_dim = 0.3 + 0.01*float(coord.y);
    float new_val = hdgl_slot(val.r, r_dim, sin(omegaTime), coord.x, coord.y);
    imageStore(latticeTex, coord, vec4(new_val,0.0,0.0,1.0));

    if(abs(new_val - prev.r) > deltaThreshold){
        uint idx = atomicAdd(count, 1);
        coords[idx] = coord;
    }

    imageStore(prevTex, coord, vec4(new_val,0.0,0.0,1.0));
}
"""

VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos + 1.0)*0.5; gl_Position = vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;

uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;

float hash_float(int i, int seed) {
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

vec3 computeVectorColor(int idx, float slot) {
    float hue = hash_float(idx,0)*360.0;
    float grad_hue = mod(hue + 180.0,360.0);
    return vec3(hue/360.0, grad_hue/360.0, slot);
}

float hdgl_slot(float val, float r_dim, float omega, int x, int y, int idx){
    float resonance = (x%4==0?0.1*sin(omegaTime*0.05 + float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5 + 0.25*sin(omegaTime*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val>threshold?1.0:0.0;
}

void main() {
    int x = int(texCoord.x * float(latticeWidth));
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    int idx = y*latticeWidth + x;
    float val = hash_float(idx,12345);
    float r_dim = 0.3 + 0.01*float(y);
    float slot = hdgl_slot(val, r_dim, sin(omegaTime), x, y, idx);
    vec3 color = computeVectorColor(idx, slot);
    fragColor = vec4(color.rgb,1.0);
}
"""

# -------------------------------
# OpenGL Init
# -------------------------------
def init_gl():
    global compute_shader, lattice_tex, prev_tex, delta_ssbo, shader
    compute_shader = compileProgram(compileShader(COMPUTE_SRC, GL_COMPUTE_SHADER))
    vs = compileShader(VERTEX_SRC, GL_VERTEX_SHADER)
    fs = compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    shader = compileProgram(vs, fs)

    # Lattice textures
    for name in ("lattice_tex", "prev_tex"):
        tex = glGenTextures(1)
        glBindTexture(GL_TEXTURE_2D, tex)
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, LATTICE_WIDTH, LATTICE_HEIGHT,
                     0, GL_RGBA, GL_FLOAT, None)
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST)
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST)
        globals()[name] = tex

    # Delta SSBO
    max_deltas = LATTICE_WIDTH * LATTICE_HEIGHT
    delta_ssbo = glGenBuffers(1)
    glBindBuffer(GL_SHADER_STORAGE_BUFFER, delta_ssbo)
    glBufferData(GL_SHADER_STORAGE_BUFFER, 4 + max_deltas*8, None, GL_DYNAMIC_DRAW)
    glBindBufferBase(GL_SHADER_STORAGE_BUFFER, 2, delta_ssbo)

    # Initialize display shader
    glUseProgram(shader)
    glUniform1fv(glGetUniformLocation(shader, "phiPowers"), len(PHI_POWERS), PHI_POWERS)
    glUniform1f(glGetUniformLocation(shader, "threshold"), THRESHOLD)
    glUniform1i(glGetUniformLocation(shader, "latticeWidth"), LATTICE_WIDTH)
    glUniform1i(glGetUniformLocation(shader, "latticeHeight"), LATTICE_HEIGHT)
    glUniform1i(glGetUniformLocation(shader, "yOffset"), 0)

# -------------------------------
# Compute Step
# -------------------------------
def compute_step_main_thread():
    global cycle, omega_time
    glUseProgram(compute_shader)
    glBindImageTexture(0, lattice_tex, 0, GL_FALSE, 0, GL_READ_WRITE, GL_RGBA32F)
    glBindImageTexture(1, prev_tex, 0, GL_FALSE, 0, GL_READ_WRITE, GL_RGBA32F)

    glUniform1f(glGetUniformLocation(compute_shader, "cycle"), cycle)
    glUniform1f(glGetUniformLocation(compute_shader, "omegaTime"), omega_time)
    glUniform1fv(glGetUniformLocation(compute_shader, "phiPowers"), 72, PHI_POWERS)
    glUniform1f(glGetUniformLocation(compute_shader, "threshold"), THRESHOLD)
    glUniform1f(glGetUniformLocation(compute_shader, "deltaThreshold"), DELTA_THRESHOLD)

    work_x = (LATTICE_WIDTH + 15) // 16
    work_y = (LATTICE_HEIGHT + 15) // 16
    glDispatchCompute(work_x, work_y, 1)
    glMemoryBarrier(GL_SHADER_STORAGE_BARRIER_BIT | GL_SHADER_IMAGE_ACCESS_BARRIER_BIT)

    # Read deltas using ctypes
    glBindBuffer(GL_SHADER_STORAGE_BUFFER, delta_ssbo)
    ptr = glMapBuffer(GL_SHADER_STORAGE_BUFFER, GL_READ_ONLY)
    if ptr:
        count = ctypes.c_uint32.from_address(ptr).value
        coords_data = (ctypes.c_uint32 * (count * 2)).from_address(ptr + 4)
        coords = np.frombuffer(coords_data, dtype=np.uint32).reshape(-1, 2)
        glUnmapBuffer(GL_SHADER_STORAGE_BUFFER)
    else:
        count = 0
        coords = np.array([], dtype=np.uint32).reshape(-1, 2)

    # Push to shared queue
    for coord in coords:
        try:
            delta_queue.put_nowait(coord)
        except queue.Full:
            break

    # PHI-weighted propagation
    propagate_to_peers(coords)

    cycle += 1.0
    omega_time += 0.05

# -------------------------------
# Audio Callback
# -------------------------------
def audio_callback(outdata, frames, time_info, status):
    samples = np.zeros((frames, 1), dtype=np.float32)
    for i in range(frames):
        try:
            x, y = delta_queue.get_nowait()
            samples[i, 0] += (x + y * LATTICE_WIDTH) / (LATTICE_WIDTH * LATTICE_HEIGHT) * 2.0 - 1.0
        except queue.Empty:
            break
    outdata[:] = np.clip(samples, -1.0, 1.0)

# -------------------------------
# GLUT Display
# -------------------------------
def display():
    global omega_time, current_yoffset
    glClear(GL_COLOR_BUFFER_BIT)
    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader, "omegaTime"), omega_time)
    yoff = manual_yoffset if manual_yoffset is not None else current_yoffset
    glUniform1i(glGetUniformLocation(shader, "yOffset"), yoff)
    # Draw fullscreen quad
    glBegin(GL_TRIANGLES)
    glVertex2f(-1, -1)
    glVertex2f(3, -1)
    glVertex2f(-1, 3)
    glEnd()
    glutSwapBuffers()
    # Auto-scroll
    if manual_yoffset is None:
        current_yoffset += AUTO_SCROLL_SPEED
        if current_yoffset > LATTICE_HEIGHT - CHUNK_HEIGHT:
            current_yoffset = 0

# -------------------------------
# GLUT Idle
# -------------------------------
def glut_idle():
    compute_step_main_thread()
    glutPostRedisplay()

# -------------------------------
# Main
# -------------------------------
def main():
    print("🚀 Starting HDGL recursive node with audio and P2P...")
    vectors = build_recursive_vectors(num_samples=32)
    export_recursive_vectors_base4096(vectors)
    export_recursive_vectors_json(vectors)
    export_binary_lattice(num_samples=LATTICE_WIDTH)

    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280, 720)
    glutCreateWindow(b"HDGL Streaming Node - Audio + P2P + Real-Time Folding")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(glut_idle)

    # Audio thread
    audio_thread = threading.Thread(target=lambda: sd.OutputStream(
        channels=1, callback=audio_callback, samplerate=SAMPLE_RATE,
        blocksize=FRAME_BUFFER_SIZE).start(), daemon=True)
    audio_thread.start()

    print("[HDGL] Streaming lattice + audio + P2P edges...")
    glutMainLoop()

if __name__ == "__main__":
    main()