#!/bin/bash
# setup_hdgl_web.sh
# Setup and run HDGL streaming service

echo "🌌 HDGL Streaming Service Setup"
echo "================================"

# Create directory structure
mkdir -p templates static music movies

# Save the HTML client
cat > templates/index.html << 'EOF'
# [The HTML content from the previous artifact goes here]
# Copy the entire HTML from the hdgl_client artifact
EOF

# Create requirements.txt
cat > requirements.txt << 'EOF'
Flask==2.3.0
Flask-CORS==4.0.0
base4096>=0.1.0
numpy>=1.24.0
EOF

# Install dependencies
echo ""
echo "📦 Installing Python dependencies..."
pip install -r requirements.txt

# Check for media directories
echo ""
echo "📁 Checking media directories..."
if [ ! -d "music" ] || [ -z "$(ls -A music)" ]; then
    echo "⚠️  'music' directory is empty. Please add .wav or .mp3 files."
fi

if [ ! -d "movies" ] || [ -z "$(ls -A movies)" ]; then
    echo "⚠️  'movies' directory is empty. Please add .mp4, .avi, or .mpeg files."
fi

# Check for export files
echo ""
echo "📊 Checking export files..."
if [ ! -f "hdgl_vectors.json" ]; then
    echo "⚠️  hdgl_vectors.json not found. Run your POC script first to generate exports."
fi

if [ ! -f "hdgl_lattice.hdgl" ]; then
    echo "⚠️  hdgl_lattice.hdgl not found. Run your POC script first to generate exports."
fi

if [ ! -f "vectors_ultrabatch.b4096" ]; then
    echo "⚠️  vectors_ultrabatch.b4096 not found. Run your POC script first to generate exports."
fi

# Create run script
cat > run_server.sh << 'EOF'
#!/bin/bash
echo "🚀 Starting HDGL Streaming Server..."
echo "Server will be available at http://localhost:5000"
echo "Press Ctrl+C to stop"
python hdgl_server.py
EOF

chmod +x run_server.sh

echo ""
echo "✅ Setup complete!"
echo ""
echo "Next steps:"
echo "1. Run your POC script (tv10_ultrabatch_gl.py) to generate export files"
echo "2. Add media files to music/ and movies/ directories"
echo "3. Start the server: ./run_server.sh"
echo "4. Open browser to http://localhost:5000"
echo ""
echo "File checklist:"
echo "  - hdgl_server.py (Flask server)"
echo "  - templates/index.html (Web client)"
echo "  - music/ (audio files)"
echo "  - movies/ (video files)"
echo "  - hdgl_vectors.json (generated by POC)"
echo "  - hdgl_lattice.hdgl (generated by POC)"
echo "  - vectors_ultrabatch.b4096 (generated by POC)"