#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
TBstream24_fixed_colors_v3.py
Full HDGL recursive node with 24 channels, streaming-safe Base4096 export with per-tile HMAC,
JSON/binary lattice export, progressive OpenGL real-time folding with auto/manual channel control,
and knobs for picture adjustment with auto-tune.
"""

import sys, json, math, struct, unicodedata, hmac, hashlib, time
import numpy as np
from base4096 import encode, decode
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 4096
LATTICE_HEIGHT = 4096
CHANNELS = 24
CHUNK_HEIGHT = LATTICE_HEIGHT // CHANNELS
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
MAX_SLOTS = 16_777_216
RECURSION_DEPTH = 3

EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BINARY = "hdgl_lattice.hdgl"
EXPORT_BASE4096 = "vectors.b4096"
HMAC_KEY = b"ZCHG-Base4096-Signature-Key"

# -------------------------------
# PICTURE ADJUSTMENT PARAMETERS
# -------------------------------
brightness = 1.0
contrast = 1.0
saturation = 1.0
gamma = 1.0
auto_tune = True

# -------------------------------
# CHAR SLOT HELPERS
# -------------------------------
def is_valid_char(c):
    try:
        cp = ord(c)
        if 0xD800 <= cp <= 0xDFFF:
            return False
        name = unicodedata.name(c)
        cat = unicodedata.category(c)
        if any(bad in name for bad in ['CONTROL','PRIVATE USE','UNASSIGNED','TAG']):
            return False
        if cat in ['Mn','Mc','Me','Cc','Cf','Cs','Cn','Co','Zs']:
            return False
        return True
    except ValueError:
        return False

def hdgl_char(idx):
    h = (idx * 2654435761) % 0x110000
    c = chr(h) if is_valid_char(chr(h)) else chr((h+1)%0x110000)
    return c

# -------------------------------
# RECURSIVE VECTORS
# -------------------------------
def unfold_slot(idx, depth=0):
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {
        "idx": idx,
        "value": val,
        "char": hdgl_char(idx),
        "children": []
    }
    if depth < RECURSION_DEPTH:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def flatten_vector(v):
    if isinstance(v, (list, tuple)):
        return ''.join(flatten_vector(x) for x in v)
    if isinstance(v, dict):
        return ''.join(flatten_vector(v[k]) for k in ['char','value','children'] if k in v)
    return str(v)

def build_recursive_vectors(num_samples=32):
    return [unfold_slot(idx) for idx in range(num_samples)]

# -------------------------------
# BASE4096 SURROGATE-SAFE
# -------------------------------
def string_to_base4096_bytes(s):
    data = bytearray()
    for c in s:
        cp = ord(c)
        if 0xD800 <= cp <= 0xDFFF or cp > 0x10FFFF:
            continue
        data.extend(cp.to_bytes(3, byteorder='big', signed=False))
    return bytes(data)

def base4096_bytes_to_string(b):
    out = []
    for i in range(0, len(b), 3):
        chunk = int.from_bytes(b[i:i+3], 'big')
        if chunk >= 0x110000:
            raise ValueError(f"Invalid codepoint {chunk}")
        out.append(chr(chunk))
    return ''.join(out)

# -------------------------------
# EXPORT FUNCTIONS
# -------------------------------
def export_recursive_vectors_base4096(vectors, outfile=EXPORT_BASE4096, hmac_key=HMAC_KEY):
    with open(outfile, "w", encoding="utf-8") as f:
        for channel in range(CHANNELS):
            h = hmac.new(hmac_key, digestmod=hashlib.sha256)
            for vector in vectors:
                combined = flatten_vector(vector)
                data = string_to_base4096_bytes(combined)
                h.update(data)
                encoded = encode(data)
                f.write(encoded + "\n")
            signature = h.digest()
            sig_encoded = encode(signature)
            f.write(f"#HMAC:{sig_encoded}\n")
    print(f"✅ Exported vectors to Base4096 with {CHANNELS} per-tile HMACs: {outfile}")

def export_recursive_vectors_json(vectors, outfile=EXPORT_JSON):
    def filter_surrogates(obj):
        if isinstance(obj, dict):
            return {k: filter_surrogates(v) for k,v in obj.items()}
        if isinstance(obj, list):
            return [filter_surrogates(x) for x in obj]
        if isinstance(obj, str):
            return ''.join(c for c in obj if 0xD800 > ord(c) or ord(c) > 0xDFFF)
        return obj
    safe_vectors = filter_surrogates(vectors)
    with open(outfile, "w", encoding="utf-8") as f:
        json.dump(safe_vectors, f, ensure_ascii=False, indent=2)
    print(f"✅ Exported {len(vectors)} vectors to JSON: {outfile}")

def export_binary_lattice(num_samples=LATTICE_WIDTH, outfile=EXPORT_BINARY):
    with open(outfile, "wb") as f:
        for idx in range(num_samples):
            val = (idx * 2654435761) % 4096 / 4096.0
            packed = struct.pack("fI", val, idx)
            f.write(packed)
    print(f"✅ Exported {num_samples} lattice slots to binary: {outfile}")

# -------------------------------
# OPENGL SHADERS
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos + 1.0)*0.5; gl_Position = vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;

uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;
uniform int channelHighlight;
uniform float brightness;
uniform float contrast;
uniform float saturation;
uniform float gamma;

float hash_float(int i, int seed) {
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

vec3 channelColors[24] = vec3[24](
    vec3(1,0,0), vec3(0,1,0), vec3(0,0,1),
    vec3(1,1,0), vec3(1,0,1), vec3(0,1,1),
    vec3(0.5,0,0), vec3(0,0.5,0), vec3(0,0,0.5),
    vec3(0.5,0.5,0), vec3(0.5,0,0.5), vec3(0,0.5,0.5),
    vec3(0.25,0,0), vec3(0,0.25,0), vec3(0,0,0.25),
    vec3(0.25,0.25,0), vec3(0.25,0,0.25), vec3(0,0.25,0.25),
    vec3(0.75,0,0), vec3(0,0.75,0), vec3(0,0,0.75),
    vec3(0.75,0.75,0), vec3(0.75,0,0.75), vec3(0,0.75,0.75)
);

vec3 applyPictureAdjustments(vec3 color){
    color = (color - 0.5)*contrast + 0.5;        // contrast
    color = color * brightness;                  // brightness
    float gray = dot(color, vec3(0.299,0.587,0.114));
    color = mix(vec3(gray), color, saturation);  // saturation
    color = pow(color, vec3(1.0/gamma));        // gamma
    return clamp(color, 0.0, 1.0);
}

float hdgl_slot(float val, float r_dim, float omega, int x, int y, int idx){
    float resonance = (x%4==0?0.1*sin(omegaTime*0.05 + float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5 + 0.25*sin(omegaTime*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val>threshold?1.0:0.0;
}

void main() {
    int x = int(texCoord.x * float(latticeWidth));
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    int idx = y*latticeWidth + x;
    float val = hash_float(idx,12345);
    float r_dim = 0.3 + 0.01*float(y);
    float slot = hdgl_slot(val, r_dim, sin(omegaTime), x, y, idx);
    vec3 color = channelColors[channelHighlight % 24] * slot;
    fragColor = vec4(applyPictureAdjustments(color),1.0);
}
"""

# -------------------------------
# OPENGL CONTROL
# -------------------------------
omega_time = 0.0
shader = None
yOffset = 0
current_channel = 0
frame_count = 0
auto_scroll = True

# -------------------------------
# OPENGL CALLBACKS
# -------------------------------
def display():
    global omega_time, yOffset
    glClear(GL_COLOR_BUFFER_BIT)
    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader, "omegaTime"), omega_time)
    glUniform1i(glGetUniformLocation(shader, "yOffset"), yOffset)
    glUniform1i(glGetUniformLocation(shader, "channelHighlight"), current_channel)
    glUniform1f(glGetUniformLocation(shader, "brightness"), brightness)
    glUniform1f(glGetUniformLocation(shader, "contrast"), contrast)
    glUniform1f(glGetUniformLocation(shader, "saturation"), saturation)
    glUniform1f(glGetUniformLocation(shader, "gamma"), gamma)
    omega_time += 0.01
    glBegin(GL_TRIANGLES)
    glVertex2f(-1,-1)
    glVertex2f( 3,-1)
    glVertex2f(-1, 3)
    glEnd()
    glutSwapBuffers()

def idle():
    global yOffset, current_channel, frame_count, omega_time, brightness, contrast, saturation, gamma
    frame_count += 1
    if auto_scroll and frame_count % 60 == 0:
        current_channel = (current_channel + 1) % CHANNELS
    yOffset = current_channel * CHUNK_HEIGHT

    # --- auto-tune adjustments ---
    if auto_tune:
        omega_time += 0.005
        brightness = 0.8 + 0.4*0.5*(1.0 + math.sin(time.time()*0.2))
        contrast   = 0.8 + 0.4*0.5*(1.0 + math.cos(time.time()*0.3))
        saturation = 0.8 + 0.4*0.5*(1.0 + math.sin(time.time()*0.15))
        gamma      = 0.9 + 0.2*0.5*(1.0 + math.cos(time.time()*0.25))

    glutPostRedisplay()

def keyboard(key, x, y):
    global current_channel, auto_scroll, brightness, contrast, saturation, gamma, auto_tune
    if key == b'w':      # previous channel
        current_channel = (current_channel - 1) % CHANNELS
        auto_scroll = False
    elif key == b's':    # next channel
        current_channel = (current_channel + 1) % CHANNELS
        auto_scroll = False
    elif key == b'a':    # toggle auto-scroll
        auto_scroll = not auto_scroll
    elif key == b'q':    # toggle auto-tune
        auto_tune = not auto_tune
    elif key == b'r':    # reset picture adjustment
        brightness = contrast = saturation = gamma = 1.0
    elif key == b'u':    # small brighten
        brightness += 0.05
    elif key == b'd':    # small darken
        brightness -= 0.05
    elif key == b'c':    # increase contrast
        contrast += 0.05
    elif key == b'x':    # decrease contrast
        contrast -= 0.05
    elif key == b'z':    # increase saturation
        saturation += 0.05
    elif key == b'e':    # decrease saturation
        saturation -= 0.05
    elif key == b'g':    # increase gamma
        gamma += 0.05
    elif key == b'h':    # decrease gamma
        gamma -= 0.05

def init_gl():
    global shader
    vs = compileShader(VERTEX_SRC, GL_VERTEX_SHADER)
    fs = compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    shader = compileProgram(vs, fs)
    glUseProgram(shader)
    loc = glGetUniformLocation(shader, "phiPowers")
    glUniform1fv(loc, len(PHI_POWERS), PHI_POWERS)
    glUniform1f(glGetUniformLocation(shader, "threshold"), THRESHOLD)
    glUniform1i(glGetUniformLocation(shader, "latticeWidth"), LATTICE_WIDTH)
    glUniform1i(glGetUniformLocation(shader, "latticeHeight"), LATTICE_HEIGHT)

# -------------------------------
# MAIN
# -------------------------------
def main():
    print("🚀 Starting HDGL recursive node with 24 fixed-color channels and picture knobs...")
    print("Controls: w/s = scroll channels, a = toggle auto-scroll, q = toggle auto-tune")
    print("r = reset, u/d = brightness, c/x = contrast, z/e = saturation, g/h = gamma")

    vectors = build_recursive_vectors(num_samples=32)
    export_recursive_vectors_base4096(vectors)
    export_recursive_vectors_json(vectors)
    export_binary_lattice(num_samples=4096)

    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL Streaming Node - 24 Channels Fixed Colors v3")

    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    glutKeyboardFunc(keyboard)
    glutMainLoop()

if __name__ == "__main__":
    main()
